/*
 * The Hefei JingTong RDC(Research and Development Centre) Group.
 * __________________
 *
 *    Copyright 2015-2017
 *    All Rights Reserved.
 *
 *    NOTICE:  All information contained herein is, and remains
 *    the property of JingTong Company and its suppliers,
 *    if any.
 */

package com.jtech.marble.util;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.util.Date;
import java.util.Locale;

/**
 * <p> 日期工具类</p>
 *
 * @author sog
 * @version 1.0
 * @since JDK 1.7
 */
@SuppressWarnings({"unused", "WeakerAccess"})
public class DateUtil {
    /**
     * 日期格式： 年-月-日
     */
    public static final String YYYYMMDD_DASH_PATTERN = "yyyy-MM-dd";

    /**
     * 日期格式： 年
     */
    public static final String YEAR_PATTERN = "yyyy";


    /**
     * 日期格式： 年月日
     */
    public static final String YYYYMMDD_PATTERN = "yyyyMMdd";


    /**
     * 日期格式： 年月日时分秒
     */
    public static final String YYYYMMDD_TIME_PATTERN = "yyyyMMdd HH:mm:ss";
    /**
     * 日期格式： 年月日时分秒毫秒
     */
    public static final String YYYYMMDD_TIMES_PATTERN = "yyyyMMdd HH:mm:ss.SSS";


    /**
     * 地区： 中国
     */
    public static final Locale DEFAULT_LOCALE = Locale.SIMPLIFIED_CHINESE;
    /**
     * 时区：正八区 / (UTC+08:00)北京，重庆，香港特别行政区，乌鲁木齐
     */
    public static final DateTimeZone DEFAULT_TIMEZONE = DateTimeZone.forOffsetHours(8);


    public static DateTimeFormatter getDateTimeFormatter(String pattern) {

        return DateTimeFormat.forPattern(pattern)
                .withZone(DEFAULT_TIMEZONE)
                .withLocale(DEFAULT_LOCALE);
    }

    /* 日期序列化*/


    public static String year(Date date) {
        return getDateTimeFormatter(YEAR_PATTERN).print(new DateTime(date));
    }

    public static String yyyyMMdd(Date date) {
        return getDateTimeFormatter(YYYYMMDD_PATTERN).print(new DateTime(date));
    }

    public static String yyyyMMddDash(Date date) {
        return getDateTimeFormatter(YYYYMMDD_DASH_PATTERN).print(new DateTime(date));
    }

    public static String yyyyMMddTime(Date date) {
        return getDateTimeFormatter(YYYYMMDD_TIME_PATTERN).print(new DateTime(date));
    }

    public static String yyyyMMddTimes(Date date) {
        return getDateTimeFormatter(YYYYMMDD_TIMES_PATTERN).print(new DateTime(date));
    }

    /*字符串转日期*/

    public static DateTime year(String dateFormat) {
        return getDateTimeFormatter(YEAR_PATTERN).parseDateTime(dateFormat);
    }

    public static DateTime yyyyMMdd(String dateFormat) {
        return getDateTimeFormatter(YYYYMMDD_PATTERN).parseDateTime(dateFormat);
    }

    public static DateTime yyyyMMddDash(String dateFormat) {
        return getDateTimeFormatter(YYYYMMDD_DASH_PATTERN).parseDateTime(dateFormat);
    }

    public static DateTime yyyyMMddTime(String dateFormat) {
        return getDateTimeFormatter(YYYYMMDD_TIME_PATTERN).parseDateTime(dateFormat);
    }

    public static DateTime yyyyMMddTimes(String dateFormat) {
        return getDateTimeFormatter(YYYYMMDD_TIMES_PATTERN).parseDateTime(dateFormat);
    }

    /*其他工具类*/


    /**
     * 获取当前时间 单位：s
     *
     * @return 当前时间
     */
    public static int unixTime() {
        return (int) (System.currentTimeMillis() / 1000);
    }

    /**
     * 通过以s为单位的当前时间返回日期
     *
     * @param unix 以s为单位的时间
     * @return unix对应的日期
     */
    public static Date dateTime(int unix) {
        return new Date(unix * 1000L);
    }

    /**
     * 通过日期返回对应的时间 单位s
     *
     * @param date 日期
     * @return 对应的以s为单位的时间
     */
    public static int unixTime(Date date) {
        return date == null ? 0 : (int) (date.getTime() / 1000);
    }

    /**
     * 通过日期返回对应的时间 单位s
     *
     * @param dateTime 日期
     * @return 对应的以s为单位的时间
     */
    public static int unixTime(DateTime dateTime) {
        return dateTime == null ? 0 : (int) (dateTime.getMillis() / 1000);
    }


    /**
     * 获取某个时间的 一天 的开始时间和结束时间
     *
     * @param time 时间
     * @return 开始和结束时间
     */
    public static Pair<DateTime, DateTime> minMaxOfDay(DateTime time) {
        final DateTime.Property millisOfDay = time.millisOfDay();
        final DateTime maximumValue = millisOfDay.withMaximumValue();
        final DateTime minimumValue = millisOfDay.withMinimumValue();
        return Pair.of(minimumValue, maximumValue);
    }

    /**
     * 获取某个时间的 一天 的开始时间和结束时间
     *
     * @param time 时间
     * @return 开始和结束时间 unixTime
     */
    public static Pair<Integer, Integer> unixMinMaxOfDay(DateTime time) {
        final DateTime.Property millisOfDay = time.millisOfDay();
        final DateTime maximumValue = millisOfDay.withMaximumValue();
        final DateTime minimumValue = millisOfDay.withMinimumValue();
        return Pair.of(unixTime(minimumValue), unixTime(maximumValue));
    }


    /**
     * 获取某个时间的 所在月份的 的开始时间和结束时间
     *
     * @param month 月
     * @return 开始和结束时间
     */
    public static Pair<DateTime, DateTime> minMaxOfMonth(DateTime month) {
        final DateTime.Property monthOfYear = month.monthOfYear();
        final DateTime maximumValue = monthOfYear.withMaximumValue();
        final DateTime minimumValue = monthOfYear.withMinimumValue();
        return Pair.of(minimumValue, maximumValue);
    }

    /**
     * 获取某个时间的 所在月份的 的开始时间和结束时间
     *
     * @param month 月
     * @return 开始和结束时间 unixTime
     */
    public static Pair<Integer, Integer> unixMinMaxOfMonth(DateTime month) {
        final DateTime.Property monthOfYear = month.monthOfYear();
        final DateTime maximumValue = monthOfYear.withMaximumValue();
        final DateTime minimumValue = monthOfYear.withMinimumValue();
        return Pair.of(unixTime(minimumValue), unixTime(maximumValue));
    }

    /**
     * 获取长整型当前时间 单位：s
     *
     * @return 以s为单位的长整型当前时间
     */
    public static long unixTimeLong() {
        return (System.currentTimeMillis() / 1000);
    }
}
