package com.jtech.marble.web;

import com.google.common.base.Strings;

import java.util.Enumeration;

import javax.servlet.http.HttpServletRequest;

import static com.jtech.marble.StringPool.AMPERSAND;
import static com.jtech.marble.StringPool.EQUALS;

/**
 * <p> request工具类</p>
 *
 * @author sog
 * @version 1.0
 * @since JDK 1.7
 */
public class RequestUtil {


    /**
     * 获取请求basePath
     */
    public static String getBasePath(HttpServletRequest request) {
        StringBuilder basePath = new StringBuilder();
        String scheme = request.getScheme();
        String domain = request.getServerName();
        int port = request.getServerPort();
        basePath.append(scheme);
        basePath.append("://");
        basePath.append(domain);
        if ("http".equalsIgnoreCase(scheme) && 80 != port) {
            basePath.append(":").append(String.valueOf(port));
        } else if ("https".equalsIgnoreCase(scheme) && port != 443) {
            basePath.append(":").append(String.valueOf(port));
        }
        return basePath.toString();
    }

    /**
     * 获取ip工具类，除了getRemoteAddr，其他ip均可伪造
     */
    public static String getIpAddr(HttpServletRequest request) {
        String ip = request.getHeader("Cdn-Src-Ip");    // 网宿cdn的真实ip
        if (ip == null || ip.length() == 0 || " unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("HTTP_CLIENT_IP");   // 蓝讯cdn的真实ip
        }
        if (ip == null || ip.length() == 0 || " unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("X-Forwarded-For");  // 获取代理ip
        }
        if (ip == null || ip.length() == 0 || " unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP"); // 获取代理ip
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP"); // 获取代理ip
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr(); // 获取真实ip
        }
        return ip;
    }

    /**
     * 移除request指定参数
     */
    public static String removeParam(HttpServletRequest request, String paramName) {
        StringBuilder queryString = new StringBuilder();
        Enumeration keys = request.getParameterNames();
        while (keys.hasMoreElements()) {
            String key = (String) keys.nextElement();
            if (key.equals(paramName)) {
                continue;
            }
            if (Strings.isNullOrEmpty(queryString.toString())) {
                queryString = new StringBuilder(key + EQUALS + request.getParameter(key));
            } else {
                queryString.append(AMPERSAND).append(key)
                        .append(EQUALS).append(request.getParameter(key));
            }
        }
        return queryString.toString();
    }


    /**
     * 判断请求是否是ajax请求
     * @param request http请求
     * @return 是否为Ajax请求
     */
    public static boolean isAjax(HttpServletRequest request){
        return request.getHeader("accept").contains("application/json")
                || (request.getHeader("X-Requested-With") != null
                && request.getHeader("X-Requested-With").contains("XMLHttpRequest"));
    }
}
